//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#include	<ctype.h>
#include	<mem.h>
#include	<alloc.h>
#include	<stdio.h>
#include	<string.h>

#include	"basics.h"
#include	"ibmkeys.h"
#include	"screen.hpp"
#include	"scrn_fcn.h"

// ----------------------------------------------------------------------------
//
//		Screen Device Functions.
//
// ----------------------------------------------------------------------------

// Static variable storage.
//
char scrnDev::overType = 0;
char *scrnDev::errorPtr = 0;

// Constructor.
//
scrnDev :: scrnDev( void )
{
	yPos = 0;
	xPos = 0;
        next = 0;
        dataCheckFcn = 0;
        preprocessFcn = 0;
}

// Virtual function.
//
void scrnDev :: show( void )
{
}

// Virtual function.
//
void scrnDev :: showCursor( void )
{
}

// Virtual function.
int scrnDev :: processChar( int c )
{
        // This should never be called.
	scrnDev::errorPtr = "Null virtual function called";
        return FAILURE;
}

// Virtual function.
int scrnDev :: validate( void )
{
        // Assume success.
	scrnDev::errorPtr = 0;
	return SUCCESS;
}

// ----------------------------------------------------------------------------
//
//		Edit Box Functions.
//
// ----------------------------------------------------------------------------

// Constructor.
//
editBox :: editBox( void )
{
	width = 0;
        cursorPos = 0;
        cnt = 0;
        buf = 0;
        filter = 0;
}

// ----------------------------------------------------------------------------
//
// Initialize edit box.
// Allocates memory and sets initial values.
//
int editBox :: init( int x , int y , int w )
{
	// Allocate memory for edit box; return if failure.
	if ( ( buf = new char [ w + 1 ] ) == 0 ) return FAILURE;
        // Clear memory buffer.
	setmem( buf , w , 0 );
        // Initialize structure.
        xPos = x;
        yPos = y;
        width = w;
        // Create null terminated string.
        buf[0] = 0;
        // And return.
        return SUCCESS;
}

// ----------------------------------------------------------------------------
//
// Displays contents of edit box.
//
void editBox :: show( void )
{
	int	i;
        char	*s;

	// Point to null at end of buffer.
        s = &buf[ cnt ];
        // Fill to end of buffer with spaces.
        i = width - cnt;
        while ( i-- ) *s++ = ' ';
        // Null terminate the string.
        *s = 0;
	// Print edit box contents.
	showStr( buf , xPos , yPos );
        // Replace terminating null.
        buf[ cnt ] = 0;
}

// ----------------------------------------------------------------------------
//
// Here to position cursor at in edit box.
//
void editBox :: showCursor( void )
{
	cursor( xPos + cursorPos , yPos );
}
//
// ----------------------------------------------------------------------------
//
// Data validation routine.
//
int editBox :: validate( void )
{
	// Return with default if no function.
        if ( dataCheckFcn == 0 )
           return scrnDev::validate();
        // Return with retults.
        return dataCheckFcn( buf );
}
//
// ----------------------------------------------------------------------------
//
// Load edit box with string.
//
void editBox :: putStr( char *s )
{
	int	len;

        // Get length of string.
        len = strlen( s );
        // Return if too long.
        if ( len > width ) return;
	// Put string into buffer.
        strcpy( buf , s );
        // Update character count.
        cnt = len;
}

// ----------------------------------------------------------------------------
//
// Inserts a character in edit box at cursor position. Moves all characters
// ahead of cursor; cursor position moves ahead 1 character.
//
void editBox :: insert( char c )
{
	char	tmp1 , tmp2;
        char	*p;
	int	i;

        // Return if we'd be trying to insert a character and buffer is full.
        if ( overType == 0 )
        {
           if ( cnt >= width ) return;
        }
        // Point to current position in string.
        p = &buf[ cursorPos ];
        // If overtype on.
        if ( overType )
        {
           // And if we're not past end of displayable characters.
           if ( cursorPos < width )
           {
              // Just replace character.
	      *p = c;
	      // If cursor at last character of line.
	      if ( cursorPos >= cnt )
	      {
		 // Update length.
		 ++cnt;
		 // Place new null character at end of string.
		 p[1] = 0;
	      }
           }
        }
        else
        {
           // Initialize temp char.
           tmp2 = c;
           // Insert character and move all characters ahead.
           while ( *p != 0 )
           {
              // Exchange characters.
              tmp1 = *p;
              *p = tmp2;
              tmp2 = tmp1;
              // Increment pointer.
              ++p;
           }
           // Add last character.
           *p++ = tmp2;
           // Add terminating null.
	   *p = 0;
	   // Update length.
	   ++cnt;
	}
	// How many characters can we print?
	i = width - cursorPos;
	// Print them.
	showStrN( &buf[ cursorPos ] , xPos + cursorPos , yPos , i );
	// Increment cursor position if not at end of line.
	if ( cursorPos < width )
        {
           ++cursorPos;
        }
        // Position cursor.
        cursor( xPos + cursorPos , yPos );
}

// ----------------------------------------------------------------------------
//
// Deletes the character to the left of the cursor. Cursor moves left one
// character, and string collapses on right.
//
void editBox :: backspace( void )
{
	char	*src , *dst;
        int	i;

	// Return if pointing to start of string.
        if ( cursorPos == 0 ) return;
        // Establish src & dst.
        src = &buf[ cursorPos ];
        dst = &buf[ cursorPos - 1 ];
        // Move characters.
        while ( *src != 0 ) *dst++ = *src++;
        // Put space where last character was.
        *dst = ' ';
        // Decrement indexes.
        --cursorPos;
	// How many characters can we print?
	i = width - cursorPos;
	// Print them.
        showStrN( &buf[ cursorPos ] , xPos + cursorPos , yPos , i );
        // Put null character where last character was.
        *dst = 0;
        // Position cursor.
        cursor( xPos + cursorPos , yPos );
        // Update length.
        --cnt;
}

// ----------------------------------------------------------------------------
//
// Moves cursor right one position in string.
// Has no effect if cursor at end of string.
//
void editBox :: rightArrow( void )
{
	// Return if pointing to null at end of string.
	if ( buf[ cursorPos ] == 0 ) return;
	// Increment cursor & display inexes.
	++cursorPos;
        // Move the cursor.
        cursor( xPos + cursorPos , yPos );
}

// ----------------------------------------------------------------------------
//
// Moves cursor left one position in string.
// Has no effect if cursor at beginning of string.
//
void editBox :: leftArrow( void )
{
	// Return if pointing to start of string.
	if ( cursorPos == 0 ) return;
	// Increment cursor & display inexes.
	--cursorPos;
	// Move the cursor.
        cursor( xPos + cursorPos , yPos );
}

// ----------------------------------------------------------------------------
//
// Edit box: process character.
// This routine processes cursor movement, character delete, and character
// insertion.
//
int editBox :: processChar( int key )
{
        int     retVal = SUCCESS;
        char    *s;

        // Special processing?
        if ( preprocessFcn != 0 )
        {
           // Call function.
           key = preprocessFcn( key );
           // Return if handled.
           if ( key == SUCCESS ) return SUCCESS;
        }
        // Which key?
        switch ( key )
        {
        // TAB: Can we leave this box?
        case HT:
           // Test data.
           if ( validate() == SUCCESS )
              // Data ok: Return with key value; let screen handle it.
              retVal = key;
           else
              // Data bad: Can't move?
              retVal = FAILURE;
           break;
        // Delete (delete forward).
        case DELETE:
           // We haven't handled it.
           retVal = key;
           break;
	// Backspace (delete backward).
        case BS:
           backspace();
           break;
	// Right arrow.
        case RIGHT_ARROW:
           rightArrow();
           break;
	// Left arrow.
        case LEFT_ARROW:
	   leftArrow();
           break;
	// Insert character.
        default:
           // If printable character.
           if ( ( key >= 32 ) && ( key <= 126 ) )
           {
              // If edit box has input filter.
              if ( filter != 0 )
              {
                 // Point to filter.
                 s = filter;
                 // Check against each character in filter.
                 while ( *s != 0 )
                 {
                    // Insert character and exit loop if match.
                    if ( key == *s++ )
                    {
                       insert( key );
                       break;
                    }
                 }
              }
              // Else just insert the character.
              else
              {
                 insert( key );
              }
	   }
           else
              // We haven't handled it.
              retVal = key;
           break;
        }
        // And return.
        return retVal;
}
//
// ----------------------------------------------------------------------------
//
//		List Box Functions.
//
// ----------------------------------------------------------------------------

// Constructor.
//
listBox :: listBox( void )
{
	width = 0;
        height = 0;
        ebCnt = 0;
        ebTop = 0;
        ebFocus = 0;
        ebLast = 0;
        ebPtr = 0;
        scroll = FALSE;
}

// ----------------------------------------------------------------------------
//
// Initializes list box.
// Allocates memory and sets initial values.
//
int listBox :: init( int x , int y , int w , int ht , int count )
{
	int	i;

	// Allocate memory for edit boxes; return if failure.
	if (  ( ebPtr = new editBox [ count ] ) == NULL ) return FAILURE;
        // Initialize structure.
        xPos = x;
        yPos = y;
	width = w;
        height = ht;
        ebCnt = count;
	// Initialize each edit box.
        // All edit boxes are initialized to display at top of list box.
	for ( i = 0 ; i < ebCnt ; ++i )
	{
	   if ( ebPtr[ i ].init( xPos , yPos , width ) != SUCCESS )
           {
              // Release storage.
              delete ebPtr;
              // And return.
	      return FAILURE;
           }
	}
	// And return.
        return SUCCESS;
}

// ----------------------------------------------------------------------------
//
// Updates list box display. Used to initially display the list box, and
// to scroll up or down through the list box.
//
void listBox :: show( void )
{
	int		i;
	editBox		*ebp;

	// Point to first edit box to be displayed in window.
	ebp = &ebPtr[ ebTop ];
        // For each line of the display.
        for ( i = 0 ; i < height ; ++i )
        {
           // Set edit box coordinates.
           ebp->yPos = yPos + i;
	   // Set cursor to start of line.
           ebp->cursorPos = 0;
           // Display edit box.
	   ebp->show();
           // Next edit box.
           ++ebp;
	}
}

// ----------------------------------------------------------------------------
//
// Here to position cursor at focused element.
//
void listBox :: showCursor( void )
{
	editBox		*ebp;

        // Point to edit box within list box.
	ebp = &ebPtr[ ebFocus ];
	// Set cursor to edit box.
	cursor( ebp->xPos + ebp->cursorPos , ebp->yPos );
}

// ----------------------------------------------------------------------------
//
// Here to insert a string at the first open edit box (at end).
//
void listBox :: addStr( char *s )
{
	int		i;
	char		tmp[ 128 ];
	editBox		*ebp;

	// Return if list box full.
	if ( ebLast >= ebCnt ) return;
	// Point to first open edit box.
	ebp = &ebPtr[ ebLast ];
	// If length of string too long.
	if ( strlen( s ) > ebp->width )
	{
	   // Copy string.
	   strcpy( tmp , s );
	   // Force to be the max size.
	   tmp[ ebp->width ] = 0;
	   // Copy to buffer.
	   strcpy( ebp->buf , tmp );
	}
	// Else just copy.
	else
	{
	   // Copy string to edit box.
	   strcpy( ebp->buf , s );
	}
	// Set number of characters in edit box.
	ebp->cnt = strlen( ebp->buf );
	// Use null filter.
	ebp->filter = 0;
	// Update index of first open edit box.
	++ebLast;
}

// ----------------------------------------------------------------------------
//
// Here to 'delete' the current line in a list box.
// This moves the contents of all 'lower' edit boxes 'up' one index.
//
void listBox :: deleteLine( void )
{
	int		moveCnt;
	editBox		*ebDst , *ebSrc;

	// Return if list box empty.
	if ( ebLast == 0 ) return;
	// Establish number of moves to make.
	moveCnt = ebLast - ebFocus - 1;
	// Point to current and next edit box.
	ebDst = &ebPtr[ ebFocus ];
	ebSrc = ebDst + 1; // &ebPtr[ ebFocus + 1 ]
	// Move all lower rows up one index.
	while ( moveCnt-- )
	{
	   // Move characters in line.
	   strcpy( ebDst->buf , ebSrc->buf );
	   // Copy character count.
	   ebDst->cnt = ebSrc->cnt;
	   // Copy filter.
	   ebDst->filter = ebSrc->filter;
	   // Increment line pointers.
	   ++ebDst , ++ebSrc;
	}
	// Erase last line.
	ebDst->buf[0] = 0;
	ebDst->cnt = 0;
	// Update index of first open edit box.
	--ebLast;
	// If we've just deleted the last line.
	if ( ebLast == ebFocus )
	{
	   // And if we're not on the first line.
	   if ( ebFocus > 0 )
	   {
	      // Move focus up one line.
	      --ebFocus;
	      // If focus is now above top of window.
	      if ( ebFocus < ebTop )
	      {
		 // If not already at top of list.
		 if ( ebTop > 0 )
		 {
		    // Move top of window.
		    --ebTop;
		 }
	      }
	   }
	}
}

// ----------------------------------------------------------------------------
//
// Here to 'insert' a blank line in a list box.
// This moves the contents of all 'lower' edit boxes 'down' one index to make
// room for blank line.
//
void listBox :: insertLine( void )
{
	int             moveCnt;
	editBox		*ebp , *ebDst , *ebSrc;

	// Return if list box full.
        if ( ebLast >= ebCnt ) return;
        // Special case: focus one past last used edit box.
        // Good to use '>=' here, because 'ebFocus' will be '1' with
        // empty file, and 'ebLast' will be '0'. This would give a count
        // of -1 below.
	if ( ebFocus >= ebLast )
        {
	   // Point to edit box with focus.
           ebp = &ebPtr[ ebFocus ];
           // Erase line.
	   ebp->buf[0] = 0;
	   ebp->cnt = 0;
           ebp->filter = 0;
	   // Move 'last' index to first open edit box.
	   ebLast = ebFocus + 1;
	   // And return.
	   return;
        }
        // Establish number of moves to make.
	moveCnt = ebLast - ebFocus;
        // Point to src & dst edit boxes for move.
        ebSrc = &ebPtr[ ebLast - 1 ];
        ebDst = ebSrc + 1; // &ebPtr[ ebLast ]
        // Move all lower rows down one index.
        while ( moveCnt-- )
	{
           // Move characters in line.
           strcpy( ebDst->buf , ebSrc->buf );
           // Copy character count.
           ebDst->cnt = ebSrc->cnt;
           // Copy filter.
           ebDst->filter = ebSrc->filter;
           // Decrement line pointers.
           --ebDst , --ebSrc;
	}
        // Erase last line ('ebDst' now must point to 'ebFocus').
        ebDst->buf[0] = 0;
        ebDst->cnt = 0;
        ebDst->filter = 0;
	// Update index of first open edit box.
        ++ebLast;
}

// ----------------------------------------------------------------------------
//
// Process character for list box.
//
int listBox :: processChar( int key )
{
        int             retVal = 0;
	int		tCursorPos , tFocus;
	int             moveCnt;
	editBox		*ebp , *ebDst , *ebSrc;

        // Special processing?
        if ( preprocessFcn != 0 )
        {
           // Call function.
           key = preprocessFcn( key );
           // Return if handled.
           if ( key == SUCCESS ) return SUCCESS;
        }
	// Which key?
	switch ( key )
	{
        //
        // TAB: Return with key value; let screen handler
        // move to next device.
        //
        case HT:
           retVal = key;
           break;
	//
	// Up arrow.
	//
	case UP_ARROW:
           // Break if at top of edit box list; we cant go up.
           if ( ebFocus == 0 ) break;
	   // Point to edit box with focus.
	   ebp = &ebPtr[ ebFocus ];
           // Save cursor column.
           tCursorPos = ebp->cursorPos;
           // If edit box with focus is at top of display window.
           if ( ebFocus == ebTop )
           {
              // Set new top edit box and edit box with focus.
	      --ebTop;
	      // Show list box.
              show();
           }
           // Point to new edit box at top of window.
           --ebp;
	   // Decrement edit box with focus.
           --ebFocus;
	   // If cursor position past end of text.
           if ( tCursorPos > ebp->cnt )
           {
              // Set cursor position to end of line.
	      tCursorPos = ebp->cnt;
           }
           // Reposition cursor.
           ebp->cursorPos = tCursorPos;
           ebp->showCursor();
           // End if.
           break;
        //
	// Down arrow.
	//
        case DOWN_ARROW:
           // Break if last edit box; we cant go down.
           if ( ebFocus >= ( ebLast - 1 ) ) break;
	   // Point to edit box with focus.
	   ebp = &ebPtr[ ebFocus ];
           // Save cursor column.
	   tCursorPos = ebp->cursorPos;
           // If edit box with focus is at bottom of display window.
           if ( ebFocus == ( ebTop + height - 1 ) )
           {
	      // Set new top edit box and edit box with focus.
              ++ebTop;
              // Show list box.
              show();
           }
           // Point to new edit box at top of window.
           ++ebp;
           // Increment edit box with focus.
	   ++ebFocus;
	   // If cursor position past end of text.
           if ( tCursorPos > ebp->cnt )
           {
              // Set cursor position to end of line.
	      tCursorPos = ebp->cnt;
	   }
           // Reposition cursor.
	   ebp->cursorPos = tCursorPos;
           ebp->showCursor();
           //
           break;
	//
        // Delete line: CTRL-Y
        //
	case CTRL_Y:
	   // Save cursor column.
	   tCursorPos = ebPtr[ ebFocus ].cursorPos;
	   // Point to edit box with focus.
	   ebp = &ebPtr[ ebFocus ];
	   // Delete line.
	   deleteLine();
	   // Show list box contents.
	   show();
	   // Point again to edit box with focus (could have changed).
	   ebp = &ebPtr[ ebFocus ];
	   // If cursor position past end of text.
	   if ( tCursorPos > ebp->cnt )
	   {
	      // Set cursor position to end of line.
	      tCursorPos = ebp->cnt;
	   }
	   // Reposition cursor.
	   ebp->cursorPos = tCursorPos;
	   ebp->showCursor();
	   //
	   break;
	//
	// Backspace; used to delete characters and lines.
	//
	case BS:
	   // Point to edit box with focus.
	   ebp = &ebPtr[ ebFocus ];
	   // If cursor about to delete first character of line.
	   if ( ebp->cursorPos == 1 )
	   {
	      // Break if more than one character on line, and first character
	      // is not a digit; entire line must be erased (from right)
	      if ( ( ebp->cnt > 1 ) && ( !isdigit( ebp->buf[0] ) ) )break;
	   }
	   // If cursor at first character of line.
	   if ( ebp->cursorPos == 0 )
	   {
	      // If no characters in line.
	      if ( ebp->cnt == 0 )
	      {
		 // Save focus.
		 tFocus = ebFocus;
		 // Delete line.
		 deleteLine();
		 // If focus hasn't changed (we haven't deleted last line).
		 if ( tFocus == ebFocus )
		 {
		    // If not at top of list.
		    if ( ebFocus != 0 )
		    {
		       // Move focus up one line.
		       --ebFocus;
		       // Now if focus above first edit box displayed.
		       if ( ebFocus < ebTop )
		       {
			  // Set new top box.
			  --ebTop;
		       }
		    }
		 }
		 // Show list box.
		 show();
		 // Point to new edit box with focus.
		 ebp = &ebPtr[ ebFocus ];
		 // Move cursor to end of line.
		 ebp->cursorPos = ebp->cnt;
		 // Reposition cursor.
		 ebp->showCursor();
	      }
	   }
	   else
	   {
	      // Let edit box process it.
              ebp->processChar( key );
	      // Clear acceptance filter if no characters in line.
	      if ( ebp->cnt == 0 ) ebp->filter = 0;
	   }
	   //
	   break;
	//
	// Carriage return.
	//
        case CR:
           // If list box is full.
           if ( ebLast >= ebCnt )
           {
              // If at last line.
              if ( ebFocus == ( ebLast - 1 ) )
              {
                 // Point to line.
                 ebp = &ebPtr[ ebFocus ];
                 // If cursor at end of line.
                 if ( ebp->cursorPos >= ebp->cnt )
                 {
                    // If scrolling enabled.
                    if ( scroll == TRUE )
                    {
                       // Move contents of all boxes up one; discard contents
                       // of top box.
                       moveCnt = ebFocus;
                       ebDst = &ebPtr[0];
                       ebSrc = &ebPtr[1];
                       // Move all rows up one index.
                       while ( moveCnt-- )
                       {
                          // Move characters in line.
                          strcpy( ebDst->buf , ebSrc->buf );
                          // Copy character count.
                          ebDst->cnt = ebSrc->cnt;
                          // Copy filter.
                          ebDst->filter = ebSrc->filter;
                          // Decrement line pointers.
                          ++ebDst , ++ebSrc;
                       }
                       // Erase last line.
                       ebp->buf[0] = 0;
                       ebp->cnt = 0;
                       ebp->filter = 0;
                       // Position cursor at beginning of line.
                       ebp->cursorPos = 0;
                       // Show list box.
                       show();
                       // Reposition cursor (at beginning of line).
                       ebp->showCursor();
                    } // Scrolling enabled.
                 } // Cursor at end of line.
              } // At last line.
              // And exit switch.
              break;
           } // List box full.
           //
	   // Point to edit box with focus.
	   ebp = &ebPtr[ ebFocus ];
	   // If at beginning of line.
	   if ( ebp->cursorPos == 0 )
	   {
	      // Insert line at current line.
              insertLine();
	      // Move down one line.
	      ++ebFocus;
	      ++ebp;
	   }
	   // Else if at end of line.
	   else if ( ebp->cursorPos >= ebp->cnt )
	   {
	      // Move down one line.
	      ++ebFocus;
	      ++ebp;
	      // Insert line.
              insertLine();
	   }
	   // Must be in middle somewhere; just break.
	   else
	   {
	      // Do nothing.
	      break;
	   }
	   // Position cursor at beginning of line.
	   ebp->cursorPos = 0;
	   // If we've moved out of display window.
	   if ( ebFocus >= ( ebTop + height ) )
	   {
	      // Update top of window index.
	      ++ebTop;
	   }
	   // Show list box.
           show();
	   // Reposition cursor (at beginning of line).
           ebp->showCursor();
	   break;
        //
	// Insert character.
        //
        default:
	   // Point to edit box with focus.
	   ebp = &ebPtr[ ebFocus ];
           // If at end of box.
           if ( ebp->cnt >= ebp->width )
           {
              // Stuff a carriage return.
              processChar( CR );
              // Adjust pointer to focused edit box.
              ebp = &ebPtr[ ebFocus ];
           }
           // Let edit box process it.
           ebp->processChar( key );
           break;
        }
        // And return.
        return retVal;
}

// ----------------------------------------------------------------------------
//
//		Screen Functions.
//
// ----------------------------------------------------------------------------

// Here to initialize a screen.
//
screen :: screen( void )
{
	scrnDev::overType = FALSE;
	count = 0;
        first = 0;
        focus = 0;
}

// ----------------------------------------------------------------------------
//
// Here to add a display element to a screen.
//
int screen :: addWidget( scrnDev *sdp )
{
	scrnDev		*tdp;

        // Make sure new device thinks it's the last one.
        sdp->next = 0;
        // Increment count.
        ++count;
	// If this is first device.
        if ( count == 1 )
        {
           // Point to first one.
           first = sdp;
           // Also gets focus.
           focus = sdp;
           // And return.
           return SUCCESS;
        }
        // Point to first device in chain.
        tdp = first;
        // Enter chain, exit at end
        for EVER
        {
           // Break if last one.
           if ( tdp->next == 0 ) break;
           // Point to next.
           tdp = tdp->next;
        }
        // Move end pointer.
        tdp->next = sdp;
        // And return.
        return SUCCESS;
}

// ----------------------------------------------------------------------------
//
// Here to refresh an entire screen.
//
void screen :: show( void )
{
        scrnDev		*sdp;

        // Point to first device.
        sdp = first;
        // Refresh each element.
	for EVER
        {
           // Exit if null pointer.
           if ( sdp == 0 ) break;
           // Refresh.
           sdp->show();
           // Next pointer.
           sdp = sdp->next;
        }
	// Show cursor for device with focus.
        focus->showCursor();
}

// ----------------------------------------------------------------------------
//
// Here to position cursor at focused element.
//
void screen :: showCursor( void )
{
        focus->showCursor();
}

// ----------------------------------------------------------------------------
//
// Here to process characters for screen.
//
int screen :: processChar ( int key )
{
        int             tmpVal , retVal = SUCCESS;
	scrnDev		*sdp;

        // Send key to device with focus.
        tmpVal = focus->processChar( key );
        // If key was handled or if there was error.
        if ( tmpVal != key )
        {
           // Return with value.
           return tmpVal;
        }
        // See if we should handle it.
        switch ( key )
        {
	//
        // Up Arrow : Go to previous screen element.
	//
        case UP_ARROW:
           // Point to first element in screen.
           sdp = first;
           // Loop through elements until you find the one that has the
           // currently focused element as 'next'.
           for EVER
           {
              // Break if you reach the end of the chain; this means that
              // the element with focus must be the first one, and we'll now
              // 'wrap' backwards to the last element.
              if ( sdp->next == 0 ) break;
              // Break when match found.
              if ( sdp->next == focus ) break;
              // Next element.
              sdp = sdp->next;
           }
           // Assign new focus.
           focus = sdp;
           // Show it.
           focus->showCursor();
           break;
	//
        // Go to next screen element if TAB or Down Arrow.
        //
        case HT:
        case DOWN_ARROW:
	   // Point to next device in chain.
           focus = focus->next;
           // Back to first if at end of chain.
           if ( focus == 0 )
	      focus = first;
	   // Show focus.
	   focus->showCursor();
	   //
           break;
        //
        // Toggle overtype if CTRL-V.
        //
        case CTRL_V:
           // Toggle overtype.
           scrnDev::overType ^= 1;
           break;
        //
        // Default: return with unhandled key.
        //
        default:
           // Return with unhandled key value; this could be an error.
           retVal = key;
	   break;
	}
	// And return.
	return retVal;
}

