//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#include	<dos.h>
#include	<conio.h>
#include	<stdio.h>
#include	"pctimer.h"
#include	"comport.h"
#include	"compiler.h"
#if		MICROSOFT_COMPILER
#include	<malloc.h>
#else
#include	<alloc.h>
#endif

// Basics.
//
#define		SUCCESS		0
#define		FAILURE		1
#define		TRUE		1
#define		FALSE		0

// Receive modes.
//
#define		RX_NORMAL	0
#define		RX_ECHO		1

// 8259 Interrupt controller data.
//
#define		PIC_EOI		0x20		// EOI port (W).
#define		PIC_IMR		0x21		// IMR port (R/W).
#define		EOI		0x20		// Normal EOI command.

#define		IRQ0_MASK	0x01		// IRQ request 0.
#define		IRQ1_MASK	0x02		// etc..
#define		IRQ2_MASK	0x04
#define		IRQ3_MASK	0x08
#define		IRQ4_MASK	0x10
#define		IRQ5_MASK	0x20
#define		IRQ6_MASK	0x40
#define		IRQ7_MASK	0x80

// Serial interrupt vectors.
//
#define		IRQ3_VECT	0x0b		// Address 0:0x2c
#define		IRQ4_VECT	0x0c		// Address 0:0x30

// 8250 Registers.
//
#define		DAT		(cp->port + 0)	// Data (R/W).
#define		DLL		(cp->port + 0)	// Divisor low.
#define		DLH		(cp->port + 1)	// Divisor high.
#define		IEN		(cp->port + 1)	// Interrupt enable.
#define		IID		(cp->port + 2)	// Interrupt ID.
#define		LCR		(cp->port + 3)	// Line control.
#define		MCR		(cp->port + 4)	// Modem control.
#define		LSR		(cp->port + 5)	// Line status.
#define		MSR		(cp->port + 6)	// Modem status.

// Interrupt ID register values.
//
#define		IID_NONE		1	// No interrupt pending.
#define		IID_MODEM_STATUS	0	// Modem status.
#define		IID_TX_EMPTY		2	// Xmit holding reg empty.
#define		IID_RX_FULL		4	// Received data available.
#define		IID_LINE_STATUS		6	// Receiver line status.

// Interrupt enable register bit masks.
//
#define		IEN_RX		0x01		// RX full.
#define		IEN_TX		0x02		// TX empty.
#define		IEN_LS		0x04		// Line status.
#define		IEN_MS		0x08		// Modem status.

// Line control register bit masks.
//
#define		LCR_7BITS	0x02		// 7 bits.
#define		LCR_8BITS	0x03		// 8 bits.
#define		LCR_2STOP	0x04		// 2 Stop bits.
#define		LCR_PEN		0x08		// Parity enable.
#define		LCR_EPS		0x10		// Even parity select.
#define		LCR_DLAB	0x80		// DLAB bit.

// Serial data format default.
//
#define		FORMAT		0x003		// 8 Data bits, 1 stop bit.

// Serial timeout count default.
//
#define		TIME_OUT	0xffff		// Timeout count

//-----------------------------------------------------------------------------
//	Global initialization status.
//-----------------------------------------------------------------------------

int	comInitStatus;

//-----------------------------------------------------------------------------
//	Local storage.
//-----------------------------------------------------------------------------

// Com port base address table.
//
static int baseAddrTable[] = { 0x3f8 , 0x2f8 , 0x3e8 , 0x2e8 };

// Com port baudrate divisor table.
//
static int baudDivisorTable[] = { 0x60 , 0x30 , 0x18 , 0x0c , 0x06 , 0x03 , 0x01 };

//-----------------------------------------------------------------------------
//	Interrupt routines.
//-----------------------------------------------------------------------------

// Pointers to 'comPort' structures; one for each interrupt source.
// These are passed to 'intService()' below.
//
static comPort	*cp3 , *cp4;

// Interrupt service; called by ISR.
//
static void intService( comPort *cp )
{
	char dchar , iidStat;

	// Loop until all interrupt sources satisfied.
	while ( ( iidStat = inportb( IID ) ) != IID_NONE )
	{

	// Depending on interrupt source...
	switch ( iidStat )
	{
	//
	// Transmitter emtpy.
	//
	case IID_TX_EMPTY:
	   // If no characters to send.
	   if ( cp->txCount == 0 )
	   {
	      // Disable transmitter.
	      outportb( IEN , inportb( IEN ) & ~IEN_TX );
	      // Clear flag.
	      cp->txRunning = 0;
	      break;
	   }
	   // Otherwise, send the next character.
	   outportb( DAT , cp->txBuf[ cp->txOut ] );
	   if ( ++cp->txOut >= cp->txSize ) cp->txOut = 0;
	   --cp->txCount;
	   break;
	//
	// Receiver full.
	//
	case IID_RX_FULL:
	   // Get character.
	   dchar = inportb( DAT );
           // If using XON/XOFF
           if ( cp->xCtrl == TRUE )
           {
              //
              // XON/XOFF characters are treated just like the CTS input.
              // Duplicating code from IID_MODEM_STATUS below.
              //
              // If XON received.
              if ( dchar == XON )
              {
                 // Break if no characters to send.
                 if ( cp->txCount == 0 ) break;
                 // Wait for holding register empty.
                 while ( ( inportb( LSR ) & LSR_TXHRE ) == 0 );
                 // Output a character.
                 outportb( DAT , cp->txBuf[ cp->txOut ] );
                 if ( ++cp->txOut >= cp->txSize ) cp->txOut = 0;
                 --cp->txCount;
                 // Enable the transmitter.
                 outportb( IEN , inportb( IEN ) | IEN_TX );
                 // Set flag.
                 cp->txRunning = 1;
                 // And break.
                 break;
              }
              else if ( dchar == XOFF )
              {
                 // Disable transmitter.
                 outportb( IEN , inportb( IEN ) & ~IEN_TX );
                 // And break.
                 break;
              }
           }
	   // Depending on receive mode...
	   switch ( cp->rxMode )
	   {
	   case RX_ECHO:
	      // Got all the characters?
	      if ( ++cp->rxIn >= cp->rxEchoCnt )
	      {
		 // Yes, turn RTS off.
		 comModemControlOff( cp , MCR_RTS );
		 // Back to normal.
		 cp->rxMode = RX_NORMAL;
		 // And reset receiver.
		 cp->rxIn = cp->rxEchoCnt = 0;
	      }
	      break;
	   case RX_NORMAL:
	   default:
	      // Put character in buffer.
	      cp->rxBuf[ cp->rxIn ] = dchar;
	      if ( ++cp->rxIn >= cp->rxSize ) cp->rxIn = 0;
	      ++cp->rxCount;
	      break;
	   }
	   break;
	//
	// Modem status inputs (cts, dsr, ri, cd).
	//
	case IID_MODEM_STATUS:
	   // Break if no change in CTS.
	   if ( ( ( cp->msrStat = inportb( MSR ) ) & MSR_DCTS ) == 0 ) break;
           // If we're using cts flow control.
           if ( cp->ctsCtrl == TRUE )
           {
              // If CTS now active.
              if ( cp->msrStat & MSR_CTS )
              {
                 // Break if no characters to send.
                 if ( cp->txCount == 0 ) break;
                 // Wait for holding register empty.
                 while ( ( inportb( LSR ) & LSR_TXHRE ) == 0 );
                 // Output a character.
                 outportb( DAT , cp->txBuf[ cp->txOut ] );
                 if ( ++cp->txOut >= cp->txSize ) cp->txOut = 0;
                 --cp->txCount;
                 // Enable the transmitter.
                 outportb( IEN , inportb( IEN ) | IEN_TX );
                 // And set flag.
                 cp->txRunning = 1;
              }
              // Else CTS now inactive.
              else
              {
                 // Disable transmitter.
                 outportb( IEN , inportb( IEN ) & ~IEN_TX );
              }
           }
	   break;
	//
	// Line status inputs.
	//
	case IID_LINE_STATUS:
	   inportb( LSR );
	   break;
	default:
	   break;
	}

	} // while interrupt sources pending.

	outportb( PIC_EOI , EOI );
}

#ifdef	__cplusplus
static void interrupt irq3(...)
#else
static void interrupt irq3()
#endif
{
	intService( cp3 );
}

#ifdef	__cplusplus
static void interrupt irq4(...)
#else
static void interrupt irq4()
#endif
{
	intService( cp4 );
}

//-----------------------------------------------------------------------------
//	Status & control routines.
//-----------------------------------------------------------------------------

// Returns number of characters in RX buffer.
//
int comInStat( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return 0;
        //
	if ( cp->irqUsed )
           return ( cp->rxCount );
        else
        {
           if ( cp->bufferChar == FALSE )
              return ( inportb( LSR ) & LSR_RXRDY );
           else
              return ( cp->rxCount );
        }
}
//
// ----------------------------------------------------------------------------
//
// Returns number of characters in TX buffer.
//
int comOutStat( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return 0;
        //
        if ( cp->irqUsed )
           return ( cp->txCount );
        else
        {
           if ( cp->bufferChar == FALSE )
           {
              if ( cp->ctsCtrl )
                 return ( ( inportb( LSR ) & LSR_TXHRE ) && ( inportb( MSR ) & MSR_CTS ) );
              else
                 return ( inportb( LSR ) & LSR_TXHRE );
           }
           else
              return ( cp->txCount );
        }
}
//
// ----------------------------------------------------------------------------
//
// Aborts a TX in process.
//
void comTxAbort( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        //
	if ( cp->irqUsed )
        {
           __asm	pushf
           __asm	cli
           outportb( IEN , inportb( IEN ) & ~IEN_TX );
           cp->txCount = cp->txIn = cp->txOut = cp->txRunning = 0;
	   __asm	popf
        }
        else
        {
           cp->txCount = cp->txIn = cp->txOut = cp->txRunning = 0;
        }
}
//
// ----------------------------------------------------------------------------
//
// Line status and modem control functions.
//
int comLineStatus( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return 0;
        //
	return inportb( LSR );
}
//
// ----------------------------------------------------------------------------
//
int comModemStatus( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return 0;
        //
	return inportb( MSR );		// Carrier detect not honored.
}
//
// ----------------------------------------------------------------------------
//
void comModemControlOn( comPort *cp , char c )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        //
	outportb( MCR , inportb( MCR ) | c );
}
//
// ----------------------------------------------------------------------------
//
void comModemControlOff( comPort *cp , char c )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        //
	outportb( MCR , inportb( MCR ) & ~(unsigned char)c );
}

//-----------------------------------------------------------------------------
//	I/O routines.
//-----------------------------------------------------------------------------

// Service com port in polled mode.
// Polls RX and puts characters in buffer.
// Polls TX ready and xmits chars in buffer.
//
void comServicePoll( comPort *cp )
{
	char	dchar;

	// Don't do it if not initialized.
	if ( cp == 0 ) return;
	// Return if not in buffered character mode.
        if ( cp->bufferChar == FALSE ) return;
        // Clear flag temporarily.
        cp->bufferChar = FALSE;
        // If character waiting to be transmitted.
        if ( cp->txCount != 0 )
        {
           // If TX buffer can accept character.
           if ( comOutStat( cp ) != 0 )
           {
              // Send character from buffer.
              outportb( DAT , cp->txBuf[ cp->txOut ] );
              if ( ++cp->txOut >= cp->txSize ) cp->txOut = 0;
              --cp->txCount;
           }
        }
        // If RX character waiting.
        if ( comInStat( cp ) != 0 )
        {
           // Get character and put in buffer.
	   dchar = inportb( DAT );
           cp->rxBuf[ cp->rxIn ] = dchar;
           if ( ++cp->rxIn >= cp->rxSize ) cp->rxIn = 0;
           ++cp->rxCount;
        }
        // Put flag back.
        cp->bufferChar = TRUE;
}

int comSendC( comPort *cp , char c )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return FAILURE;
        //
	if ( cp->irqUsed )
        {
           __asm pushf
           __asm cli
           if ( cp->txCount == cp->txSize )
           {
              __asm	popf
              return FAILURE;
           }
           if ( cp->txRunning )
           {
              cp->txBuf[ cp->txIn ] = c;
              if ( ++cp->txIn >= cp->txSize ) cp->txIn = 0;
              ++cp->txCount;
           }
           else
           {
              while ( ( inportb( LSR ) & LSR_TXHRE ) == 0 );
              outportb( DAT , c );
              outportb( IEN , inportb( IEN ) | IEN_TX );
              cp->txRunning = 1;
           }
           __asm popf
           return SUCCESS;
        }
        else
        {
           if ( cp->bufferChar == FALSE )
           {
              while ( comOutStat( cp ) == 0 );
              outportb( DAT , c );
           }
           else
           {
              cp->txBuf[ cp->txIn ] = c;
              if ( ++cp->txIn >= cp->txSize ) cp->txIn = 0;
              ++cp->txCount;
           }
           return SUCCESS;
        }
}
//
// ----------------------------------------------------------------------------
//
int comRecv( comPort *cp )
{
	char	dchar;

	// Don't do it if not initialized.
	if ( cp == 0 ) return NULL_CHAR;
        //
	if ( cp->irqUsed )
        {
           if ( cp->rxCount == 0 ) return NULL_CHAR;
           __asm pushf
           __asm cli
           dchar = cp->rxBuf[ cp->rxOut ];
           if ( ++cp->rxOut >= cp->rxSize ) cp->rxOut = 0;
           --cp->rxCount;
           __asm popf
           return (unsigned char)dchar;
        }
        else
        {
           // If using standard polling technique.
           if ( cp->bufferChar == FALSE )
           {
              while ( comInStat( cp ) == 0 );
              return (unsigned char)inportb( DAT );
           }
           // Else buffer mode.
           else
           {
              // Same code as in 'comRecv()' for interrupt mode, except
              // that we don't disable interrupts.
              if ( cp->rxCount == 0 ) return NULL_CHAR;
              dchar = cp->rxBuf[ cp->rxOut ];
              if ( ++cp->rxOut >= cp->rxSize ) cp->rxOut = 0;
              --cp->rxCount;
              return (unsigned char)dchar;
           }
        }
}
//
// ----------------------------------------------------------------------------
//
// Here to read a single character from the com port.
// Returns TIMEOUT (-1) if no character received within time limit specified (seconds).
//
int comRecvT( struct comPort *cp , int timeout )
{
	int		dchar;

	// Don't do it if not initialized.
	if ( cp == 0 ) return NULL_CHAR;
        // Start timer.
        tmrStartSec( cp->tmr , timeout );
        // Wait for character or timeout.
        for (;;)
        {
           // Return if timer times out.
           if ( tmrFlag( cp->tmr ) ) return TIMEOUT;
           // Query serial port.
           dchar = comRecv( cp );
           // Break if character received.
           if ( dchar != -1 ) return dchar;
        }
}
//
// ----------------------------------------------------------------------------
//
// Here to monitor the receive buffer until a specified number of characters
// have been received.
// Returns SUCCESS if at least 'cnt' characters have been received.
// Returns TIMEOUT (-1) if the specified number of characters have not been
// received within time limit specified (seconds).
//
int comRecvTC( struct comPort *cp , int timeout , int cnt )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return NULL_CHAR;
        // Start timer.
        tmrStartSec( cp->tmr , timeout );
        // Wait for character.
        for (;;)
        {
           // Return if timer times out.
           if ( tmrFlag( cp->tmr ) ) return TIMEOUT;
           // Query serial port.
	   if ( comInStat( cp ) >= cnt ) return SUCCESS;
        }
}

//
// ----------------------------------------------------------------------------
//
void comUngetC( comPort *cp , char c )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        //
	__asm pushf
	__asm cli
	if ( --cp->rxOut < 0 ) cp->rxOut = cp->rxSize - 1;
	cp->rxBuf[ cp->rxOut ] = c;
	++cp->rxCount;
	__asm popf
}
//
// ----------------------------------------------------------------------------
//
// Here to flush rx buffer.
//
void comFlushRx( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        //
	__asm pushf
	__asm cli
        cp->rxOut = cp->rxIn = 0;
        cp->rxCount = 0;
	__asm popf
}
//
// ----------------------------------------------------------------------------
//
// Other sending routines common to both interrupt and non-interrupt
// modes.
//
// Send a null terminated string 's'.
//
int comSendS( comPort *cp , char *s )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return FAILURE;
        //
	while ( *s )
	{
	   if ( comSendC( cp , *s++ ) != SUCCESS ) return FAILURE;
	}
	return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
// Send a binary string 's' which has 'cnt' elements.
//
int comSendSI( comPort *cp , char *s , int cnt )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return FAILURE;
        //
	while ( cnt-- )
	{
	   if ( comSendC( cp , *s++ ) != SUCCESS ) return FAILURE;
	}
	return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
// Here to send a string out an RS485 two wire setup.
// This sets up the receiver to be in ECHO mode.
//
int comSend485( comPort *cp , char *s , int cnt )
{
	int	status;

	// Don't do it if not initialized.
	if ( cp == 0 ) return FAILURE;
	// Reset receiver, and set for echo.
	__asm pushf
	__asm cli
	cp->rxCount = cp->rxIn = cp->rxOut = 0;
	cp->rxMode = RX_ECHO;
	cp->rxEchoCnt = cnt;
	__asm popf
	// Turn on the RTS line.
	comModemControlOn( cp , MCR_RTS );
	// Send the string.
	status = comSendSI( cp , s , cnt );
	// Reset recevier if error.
	if ( status != SUCCESS )
	{
	   // Reset receive mode.
	   cp->rxMode = RX_NORMAL;
	   // Turn RTS off.
	   comModemControlOff( cp , MCR_RTS );
	}
	// Return with status.
	return status;
}

//-----------------------------------------------------------------------------
//	Initialization.
//-----------------------------------------------------------------------------

static void clearRegs( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        // Start timer.
        tmrStartSec( cp->tmr , 1 );
        // Loop until done.
	for (;;)
	{
           //Break if timer times out.
           if ( tmrFlag( cp->tmr ) ) break;
           // Input from registers.
	   inportb( LSR );
	   inportb( DAT );
	   inportb( MSR );
	   inportb( IID );
           // Until proper IID value.
	   if ( inportb( IID ) == 1 ) break;
	}
        // Stop timer.
        tmrStop( cp->tmr );
}
//
// ----------------------------------------------------------------------------
//
void comSetBaud( comPort *cp , int baudRate )
{
	int	i , lcrStat;

	// Don't do it if not initialized.
	if ( cp == 0 ) return;
	// Save LCR status.
	lcrStat = inportb( LCR );
	// Set DLAB.
	outportb( LCR , 0x80 | lcrStat );
	// Get divisor.
	i = baudDivisorTable[ baudRate ];
	// Load baudrate divisor.
	outportb( DLL , i & 0x00ff );
	outportb( DLH , i >> 8 );
	// Restore LCR status.
	outportb( LCR , lcrStat );
}
//
// ----------------------------------------------------------------------------
//
void comSetFormat( comPort *cp , int format )
{
	char	dFmt = 0;

	// Don't do it if not initialized.
	if ( cp == 0 ) return;
	// Only handles 7 or 8 bits.
	if ( format & _7BIT )
	   dFmt = LCR_7BITS;
	else
	   dFmt = LCR_8BITS;
	// Number of stop bits.
	if ( format & _2STOP )
	   dFmt |= LCR_2STOP;
	// Parity control.
	if ( format & PARITY )
	{
	   dFmt |= LCR_PEN;
	   if ( format & EVEN )
	      dFmt |= LCR_EPS;
	}

	// Apparently, I'm not worried about 'stick parity' or 'set break'.

	// Output to LCR.
	outportb( LCR , dFmt );
}
//
// ----------------------------------------------------------------------------
//
// Full serial port initialization routine.
//
comPort *comInit( int irqOn , int port , int format , int baudRate , int rxSize , int txSize )
{
	char	picImrMask = 0xff;
        comPort	*cp;

	// Assume success.
	comInitStatus = SUCCESS;
        // Check port number.
        if ( ( port < COM1 ) || ( port > COM4 ) )
        {
           // Exit with error.
           comInitStatus = INIT_BADPORT;
           goto ExitHere;
        }
	// Allocate memory for 'comPort'.
	if ( ( cp = (comPort *)malloc( sizeof( struct comPort ) ) ) == NULL )
	{
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
	// Allocate memory for timer.
	if ( ( cp->tmr = tmrInit() ) == NULL )
	{
	   // Release previously allocated storage.
	   free( cp );
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
	// Allocate memory for rx buffer.
	if ( ( cp->rxBuf = (char *)malloc( sizeof( char ) * rxSize ) ) == NULL )
	{
	   // Release previously allocated storage.
           free( cp->tmr );
	   free( cp );
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
	// Allocate memory for tx buffer.
	if ( ( cp->txBuf = (char *)malloc( sizeof( char ) * txSize ) ) == NULL )
	{
	   // Release previously allocated storage.
           free( cp->tmr );
	   free( cp->rxBuf );
	   free( cp );
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
        // Initialize timer.
        cp->tmr = tmrInit();
	// Initialize data areas.
        cp->initPoll = FALSE;
        cp->xCtrl = FALSE;
        cp->ctsCtrl = FALSE;
	cp->port = baseAddrTable[ port ];
	cp->rxCount = cp->txCount = cp->rxIn = cp->txIn = cp->rxOut = cp->txOut = 0;
	cp->rxMode = RX_NORMAL;
	cp->txMode = 0;
	cp->txRunning = 0;
        cp->bufferChar = 0;
	cp->irqUsed = irqOn;
	cp->rxSize = rxSize;
	cp->txSize = txSize;
	// Disable 8250 interrupts.
	// Zero line control register.
	// Turn modem control lines off.
	outportb( IEN , 0 );
	outportb( LCR , 0 );
	outportb( MCR , 0 );
	// Set baud rate & data format.
	comSetBaud( cp , baudRate );
	comSetFormat( cp , format );
	// Clear registers.
	clearRegs( cp );
	//If using interrupt mode.
        if ( cp->irqUsed )
        {
           // Set interrupt vectors and sources.
           switch ( cp->port )
           {
           case 0x3f8:
	   case 0x3e8:
              cp4 = cp;
              cp->oldHandler = getvect( IRQ4_VECT );
              setvect( IRQ4_VECT , irq4 );
              picImrMask &= ~IRQ4_MASK;
              break;
           case 0x2f8:
	   case 0x2e8:
              cp3 = cp;
              cp->oldHandler = getvect( IRQ3_VECT );
              setvect( IRQ3_VECT , irq3 );
              picImrMask &= ~IRQ3_MASK;
              break;
           default:
              break;
           }
           // Enable all sources of interrupts; modem status changes will
           // only be acted on if cts flow control used.
           outportb( IEN , IEN_RX | IEN_LS | IEN_MS );
           outportb( MCR, MCR_IEN );
	}
        else
        {
           outportb( IEN , 0 );
        }
	// Clear registers, again.
	clearRegs( cp );
	// Set modem status register flags.
	cp->msrStat = inportb( MSR );
	//If using interrupts.
        if ( cp->irqUsed )
        {
           // Enable com interrupts on 8259.
           //
           cp->oldImrMask = inportb( PIC_IMR );
           outportb( PIC_IMR , cp->oldImrMask & picImrMask );
           outportb( PIC_EOI , EOI );
	}
ExitHere:
	// If successful.
	if ( comInitStatus == SUCCESS )
           // Return with pointer.
           return cp;
        // Else return with null pointer.
        else
           return NULL;
}
//
// ----------------------------------------------------------------------------
//
// Initialization routine that sets port number & pointers, but does not
// access uart.
// Assumes that serial port is already set up, and you just want to access it.
// Interrupts can't be used.
//
comPort *comInitPoll( int port , int rxSize , int txSize )
{
	comPort		*cp;

	// Assume success.
	comInitStatus = SUCCESS;
        // Check port number.
        if ( ( port < COM1 ) || ( port > COM4 ) )
        {
           // Exit with error.
           comInitStatus = INIT_BADPORT;
           goto ExitHere;
	}
	// Allocate memory for 'comPort'.
	if ( ( cp = (comPort *)malloc( sizeof( struct comPort ) ) ) == NULL )
	{
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
	// Allocate memory for timer.
	if ( ( cp->tmr = tmrInit() ) == NULL )
	{
	   // Release previously allocated storage.
	   free( cp );
	   // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
	// Allocate memory for rx buffer.
	if ( ( cp->rxBuf = (char *)malloc( sizeof( char ) * rxSize ) ) == NULL )
	{
	   // Release previously allocated storage.
           free( cp->tmr );
	   free( cp );
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
	// Allocate memory for tx buffer.
	if ( ( cp->txBuf = (char *)malloc( sizeof( char ) * txSize ) ) == NULL )
	{
	   // Release previously allocated storage.
           free( cp->tmr );
	   free( cp->rxBuf );
	   free( cp );
           // Exit with error.
	   comInitStatus = INIT_MALLOC;
	   goto ExitHere;
	}
        // Initialize timer.
        cp->tmr = tmrInit();
	// Initialize data areas.
	cp->initPoll = TRUE;
	cp->xCtrl = FALSE;
	cp->ctsCtrl = FALSE;
	cp->port = baseAddrTable[ port ];
	cp->rxCount = cp->txCount = cp->rxIn = cp->txIn = cp->rxOut = cp->txOut = 0;
	cp->rxMode = RX_NORMAL;
	cp->txMode = 0;
	cp->txRunning = 0;
	cp->bufferChar = 0;
	cp->irqUsed = 0;
	cp->rxSize = rxSize;
	cp->txSize = txSize;
ExitHere:
	// If successful.
	if ( comInitStatus == SUCCESS )
	   // Return with pointer.
           return cp;
        // Else return with null pointer.
        else
           return NULL;
}
//
// ----------------------------------------------------------------------------
//
void comDestroy( comPort *cp )
{
	// Don't do it if not initialized.
	if ( cp == 0 ) return;
        // Don't mess with chip if in polled mode.
        if ( cp->initPoll == TRUE ) goto ReleaseMemory;
	// Disable 8250 interrupts.
	// Zero line control register.
	// Turn modem control lines off.
	outportb( IEN , 0 );
	outportb( LCR , 0 );
	outportb( MCR , 0 );
	// Clear registers.
	clearRegs( cp );
	// If using interrupts.
        if ( cp->irqUsed )
        {
           // Put old interrupt handler back.
           switch ( cp->port )
           {
           case 0x3f8:
           case 0x3e8:
              setvect( IRQ4_VECT , cp->oldHandler );
              break;
           case 0x2f8:
           case 0x2e8:
              setvect( IRQ3_VECT , cp->oldHandler );
              break;
           default:
              break;
           }
           // Put old PIC IMR mask back.
           outportb( PIC_IMR , cp->oldImrMask );
        }
ReleaseMemory:
 	// Release storage.
        free( cp->tmr );
	free( cp->txBuf );
	free( cp->rxBuf );
	free( cp );
}
