/**
 * Copyright 2013, Contemporary Control Systems, Inc.
 */

/******************************************************************************
 *
 * Form Object; now based on jQuery functions.
 *
 ******************************************************************************/

/**
 * @type {{None: number, Button: number, CheckItem: number, EditBox: number, ComboBox: number, Label: number, TextArea: number}}
 */
var fmoType =
{
    None		: 0,
    Button		: 1,
    CheckItem	: 2,
    EditBox		: 3,
    ComboBox	: 4,
    Label		: 5,
    TextArea	: 6,
	Image		: 7,
	Div			: 8
};

/**
 *
 * @param {number} type
 * @param {string} dom_element_id
 * @param {...*} argv
 * @constructor
 *
 * argv[2] = FormObject array
 */
function FormObject(type , dom_element_id , argv)
{
    "use strict";

    this.jqObj = $('#'+dom_element_id);
    if ((this.jqObj.length === 0) || (this.jqObj.length > 1))
    {
        if (this.jqObj.length === 0)
             alert("Element not found: " + dom_element_id);
        else
            alert("Multiple elements found: " + dom_element_id);
        this.jqObj = null;
        return;
    }

    /** @type {number} */
    this.type = type;
    /** @type {boolean} */
    this.dirty = false;
    /** @type {boolean} */
    this.hidden = false;
    /** @type {string} */
    this.xtag = '';

    this.rdomEle = null;            // _DBUG: soon to be obsoleted by path methods.

    /** @type {jQuery|{left: number, top: number}} */
    this.pos = this.jqObj.offset();

    // If we're to add this to an array of form objects.
    if (arguments.length === 3)
    {
        arguments[2].push(this);
    }
}

/******************************************************************************
 * Get and set methods.
 */

/**
 * @return {string}
 */
FormObject.prototype.GetValue = function ()
{
    /** @type {string | jQuery} */
    var retval = '';

    switch (this.type)
    {
    case fmoType.CheckItem:
        if ($(this.jqObj[0]).attr('checked'))
            retval = '1';
        else
            retval = '0';
        break;
    case fmoType.Label:
        retval = this.jqObj.find('*').last().html().toString();
        break;
	case fmoType.Image:
		retval = this.jqObj.find('img').attr('src');
		break;
    default:
        retval = this.jqObj.val().toString();
        break;
    }
    return retval;
};

/**
 *
 * @param {*} value
 */
FormObject.prototype.SetValue = function(value)
{
    /** @type {boolean} */
    var bVal;

    switch (this.type)
    {
    case fmoType.CheckItem:
        if      (value === true)
            bVal = true;
        else if (value === false)
            bVal = false;
        else
            bVal = (value !== '0');
        $(this.jqObj[0]).attr('checked' , bVal);
        break;
    case fmoType.Label:
        this.jqObj.find('*').last().html(value.toString());
        break;
	case fmoType.Image:
		this.jqObj.find('img').attr('src' , value.toString());
		break;
    default:
        this.jqObj.val(value.toString());
        break;
    }
};

/**
 * @param {string} color
 */
FormObject.prototype.SetColor = function (color)
{
	if (this.type === fmoType.Label)
	{
		this.jqObj.find('span').css('color', color);
	}
};

/******************************************************************************
 * Basic methods.
 */

/**
 * @returns {string|boolean|!jQuery|*}
 */
FormObject.prototype.GetID = function ()
{
    return this.jqObj.prop('id');
};

/**
 * @param {boolean} t_f
 */
FormObject.prototype.SetDisabled = function (t_f)
{
    this.jqObj.prop('disabled' , t_f);

//    this.jqObj[0].disabled = t_f;
};

/**
 * @returns {boolean}
 */
FormObject.prototype.GetHidden = function()
{
    return this.hidden;
};

/**
 * @param {boolean} t_f
 */
FormObject.prototype.SetHidden = function (t_f)
{
    if (t_f)
    {
        this.hidden = true;
        this.jqObj.hide();
    }
    else
    {
        this.hidden = false;
        this.jqObj.show();
    }
};

/**
 * @returns {Object.<number, number>}
 */
FormObject.prototype.GetPosition = function ()
{
    return this.jqObj.offset();
};

/**
 * @param {number | null} left
 * @param {number | null} top
 */
FormObject.prototype.SetPosition = function (left, top)
{
    /** @type {jQuery | {top:number , left:number}} */
    var next_pos = this.jqObj.offset();

    if (left !== null)
        next_pos.left = left;
    if (top !== null)
        next_pos.top = top;
    this.jqObj.offset({ top: next_pos.top , left: next_pos.left });
};

/**
 * Sets the "hover text" when the mouse is over the element.
 *
 * @param {string} hover_text
 */
FormObject.prototype.SetHoverText = function(hover_text)
{
    this.jqObj.prop('title' , hover_text);

//    this.jqObj[0].title = hover_text;
};

/**
 * @param {boolean} t_f
 */
FormObject.prototype.SetResize = function(t_f)
{
    if (t_f)
        this.jqObj.css('resize' , 'both');
    else
        this.jqObj.css('resize' , 'none');
};

/******************************************************************************
 * Handler setting methods.
 */

/**
 * Sets the "onclick" handler for this form object. Handlers are passed 'this.id'
 * for identification.
 *
 * @param handler
 */
FormObject.prototype.SetOnclickHandler = function(handler)
{
    this.jqObj.click(function()
    {
        return handler(this.id);
    });
};

/**
 * Sets the "onkeyup" handler for this form object. Handlers are passed 'this.id'
 * for identification.
 *
 * @param handler
 */
FormObject.prototype.SetOnkeyupHandler = function(handler)
{
    this.jqObj.keyup(function()
    {
        return handler(this.id);
    });
};

/**
 * Sets the "onchange" handler for this form object. Handlers are passed 'this.id'
 * for identification.
 *
 * @param handler
 */
FormObject.prototype.SetOnchangeHandler = function(handler)
{
    this.jqObj.change(function()
    {
        return handler(this.id);
    });
};

/**
 * @param {function(string)} handler
 */
FormObject.prototype.SetOnFocusInHandler = function (handler)
{
	this.jqObj.focus(function ()
	{
		return handler(this.id);
	});
};

/**
 * @param {function(string)} handler
 */
FormObject.prototype.SetOnFocusOutHandler = function (handler)
{
	this.jqObj.focusout(function ()
	{
		return handler(this.id);
	});
};

/******************************************************************************
 * Edit box.
 */

/**
 * @param {number} max_length
 */
FormObject.prototype.SetMaxLength = function (max_length)
{
    if (this.type !== fmoType.EditBox)
        return;
    this.jqObj.prop('maxLength' , max_length);
};

/******************************************************************************
 * Combo box.
 */

/**
 * Clears the combo box before reusing it.
 */
FormObject.prototype.Clear = function ()
{
    var ele = this.jqObj[0];

    if (this.type !== fmoType.ComboBox)
        return;

    while (ele.options.length > 0)
        ele.remove(0);
    ele.selectedIndex = 0;
};

/**
 * Adds a new option to the combo box.
 *
 * @param {string} text
 * @param {string} value
 */
FormObject.prototype.AddOpt = function (text, value)
{
    var opt;

    if (this.type !== fmoType.ComboBox)
        return;

    opt = '<option value="' + value + '">' + text + '</option>';
    this.jqObj.append($(opt));
};

/**
 * Adds options to a combo box from an array of strings.
 * The string values are both the display text and value text.
 *
 * @param text_array
 */
FormObject.prototype.AddArray1D = function (text_array)
{
    var i;

    if (this.type !== fmoType.ComboBox)
        return;

    for (i = 0; i < text_array.length; ++i)
    {
        this.AddOpt(text_array[i], text_array[i]);
    }
};

/**
 * Adds option to a combo box from an array of string pairs.
 * The pairs are the display text (0), and the value text [1].
 *
 * @param pair_array
 */
FormObject.prototype.AddArray2D = function (pair_array)
{
    var i;

    if (this.type !== fmoType.ComboBox)
        return;

    for (i = 0; i < pair_array.length; ++i)
    {
        this.AddOpt(pair_array[i][0], pair_array[i][1]);
    }
};

/******************************************************************************
 * Labels.
 */

/**
* @param {string} link
*/
FormObject.prototype.SetLink = function(link)
{
    $(this.jqObj[0]).find('a').attr('href' , link);
};

/******************************************************************************
 * Buttons.
 */

FormObject.prototype.SetButtonText = function(new_text)
{
    $(this.jqObj).attr('value' , new_text);
};

/******************************************************************************
 * Exports.
 */

window['FormObject'] = FormObject;

FormObject.prototype['GetID'] = FormObject.prototype.GetID;
FormObject.prototype['SetDisabled'] = FormObject.prototype.SetDisabled;
FormObject.prototype['SetHidden'] = FormObject.prototype.SetHidden;
FormObject.prototype['SetPosition'] = FormObject.prototype.SetPosition;
FormObject.prototype['GetPosition'] = FormObject.prototype.GetPosition;
FormObject.prototype['SetOnclickHandler'] = FormObject.prototype.SetOnclickHandler;
FormObject.prototype['SetOnkeyupHandler'] = FormObject.prototype.SetOnkeyupHandler;
FormObject.prototype['SetOnchangeHandler'] = FormObject.prototype.SetOnchangeHandler;
FormObject.prototype['SetHoverText'] = FormObject.prototype.SetHoverText;
FormObject.prototype['SetResize'] = FormObject.prototype.SetResize;
FormObject.prototype['GetValue'] = FormObject.prototype.GetValue;
FormObject.prototype['SetValue'] = FormObject.prototype.SetValue;

FormObject.prototype['SetMaxLength'] = FormObject.prototype.SetMaxLength;

FormObject.prototype['Clear'] = FormObject.prototype.Clear;
FormObject.prototype['AddOpt'] = FormObject.prototype.AddOpt;
FormObject.prototype['AddArray1D'] = FormObject.prototype.AddArray1D;
FormObject.prototype['AddArray2D'] = FormObject.prototype.AddArray2D;

FormObject.prototype['SetLink'] = FormObject.prototype.SetLink;
